#ifndef __CameraLinkCmd_h__
#define __CameraLinkCmd_h__

/*
	Windows: x64, Win7 or above
	Linux: x64, GLIBC 2.17 or above
*/

#ifndef CAMLINK_API
#if defined(_WIN32)
#ifdef CAMERALINKCMD_EXPORTS
#define CAMLINK_API(x) __declspec(dllexport) x __cdecl
#else
#define CAMLINK_API(x) __declspec(dllimport) x __cdecl
#endif
#else
#define CAMLINK_API(x) x
#define __stdcall
#endif
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifndef TDIBWIDTHBYTES
#define TDIBWIDTHBYTES(bits)  ((unsigned)(((bits) + 31) & (~31)) / 8)
#endif

typedef struct { int unused; } *HCamL;

/*
* Open/Close COM
* protNum:
*		Windows: COM Port Number, 0 => COM0, 1 => COM1, 2 => COM2, ...
*		Linux: Camera Link card index, 0, 1, 2, ...
* baudRate <= 0 means to use default (115200)
*/
CAMLINK_API(HCamL) CamL_Init(int portNum, int baudRate);
CAMLINK_API(void) CamL_UnInit(HCamL h);


typedef void (__stdcall* CameraLinkCmd_CMD_CALLBACK)(char* strCmd, void* pCallbackCtx);
CAMLINK_API(bool)  CamL_WriteCmdCallback(HCamL h, CameraLinkCmd_CMD_CALLBACK function, void* pCallbackCtx);
CAMLINK_API(bool)  CamL_DeleteCmdCallback(HCamL h);

/*
* FPGA Version
*/
CAMLINK_API(bool) CamL_GetFpgaVersion(HCamL h, char fpgaVer[16], char fpgaDate[16]);

/*
* Model
*/
CAMLINK_API(bool) CamL_GetModel(HCamL h, char mn[32]);

/*
* Product Number
*/
CAMLINK_API(void) CamL_SetPN(HCamL h, char pn[64]);
CAMLINK_API(bool) CamL_GetPN(HCamL h, char pn[64]);

/*
* Series Number
*/
CAMLINK_API(void) CamL_SetSN(HCamL h, char sn[64]);
CAMLINK_API(bool) CamL_GetSN(HCamL h, char sn[64]);

/*
* Auto Exposure
*/
CAMLINK_API(void) CamL_EnableAutoExposure(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsAutoExposureEnabled(HCamL h);

/*
* Conversion Gain
*/
CAMLINK_API(void) CamL_SetCG(HCamL h, int cg);
CAMLINK_API(int)  CamL_GetCG(HCamL h);
CAMLINK_API(void) CamL_GetCGRange(HCamL h, int* pMin, int* pMax, int* pDef);

/*
* Exposure Read Mode
* mode: 0 = ITR, 1 = IWR
*/
CAMLINK_API(bool) CamL_IsSupportExpoReadMode(HCamL h);
CAMLINK_API(void) CamL_SetExpoReadMode(HCamL h, int mode);
CAMLINK_API(int)  CamL_GetExpoReadMode(HCamL h);

/*
* Exposure Time
* Unit: us
*/
CAMLINK_API(void) CamL_SetExpoTime(HCamL h, unsigned time);
CAMLINK_API(unsigned)  CamL_GetExpoTime(HCamL h);
CAMLINK_API(void) CamL_GetExpoTimeRange(HCamL h, unsigned* pMin, unsigned* pMax, unsigned* pDef);

/*
* Sequence Exposure Time
* Unit: us
*/
CAMLINK_API(bool) CamL_IsSupportSeqExpoTime(HCamL h);
CAMLINK_API(void) CamL_EnableSeqExpoTime(HCamL h, bool bEnabled);
CAMLINK_API(bool) CamL_IsSeqExpoTimeEnabled(HCamL h);
CAMLINK_API(void) CamL_SetSeqExpoTimeCount(HCamL h, unsigned count);
CAMLINK_API(unsigned)  CamL_GetSeqExpoTimeCount(HCamL h);
CAMLINK_API(void) CamL_SetSeqExpoTime(HCamL h, int idx, unsigned time);
CAMLINK_API(unsigned)  CamL_GetSeqExpoTime(HCamL h, int idx);

/*
* Exposure Gain
* Unit: us
*/
CAMLINK_API(void) CamL_SetExpoGain(HCamL h, unsigned gain);
CAMLINK_API(unsigned)  CamL_GetExpoGain(HCamL h);
CAMLINK_API(void) CamL_GetExpoGainRange(HCamL h, unsigned* pMin, unsigned* pMax, unsigned* pDef);

/*
* Frame Rate
*/
CAMLINK_API(void) CamL_SetFrameRate(HCamL h, int frameRate);
CAMLINK_API(int)  CamL_GetFrameRate(HCamL h);
CAMLINK_API(int)  CamL_GetMaxFrameRate(HCamL h);

/*
* ROI
* xOffset:         Must be a multiple of 8
* width:           Must be a multiple of 8, and be greater than or equal to 16
* height:          Must be greater than or equal to 4
* SetRoi(0,0,0,0): Close ROI
*/
CAMLINK_API(void) CamL_GetSize(HCamL h, int* pWidth, int* pHeight);
CAMLINK_API(void) CamL_SetRoi(HCamL h, int xOffset, int yOffset, int width, int height);
CAMLINK_API(void) CamL_GetRoi(HCamL h, int *pxOffset, int *pyOffset, int* pWidth, int* pHeight);

/* 
* multiple Roi 
* xOffset:         Must be a multiple of 8
* xWidth:          Must be a multiple of 8
* yOffset:         Must be a multiple of 16
* yHeight:         Must be a multiple of 16
*/
CAMLINK_API(bool) CamL_IsSupportRoiN(HCamL h);
CAMLINK_API(void) CamL_SetRoiN(HCamL h, unsigned xOffset[], unsigned yOffset[], unsigned xWidth[], unsigned yHeight[], unsigned hNum, unsigned vNum);
CAMLINK_API(void) CamL_GetRoiN(HCamL h, unsigned xOffset[], unsigned yOffset[], unsigned xWidth[], unsigned yHeight[], unsigned* hNum, unsigned* vNum);

#if defined(_WIN32)
typedef void (__cdecl* CAML_PROGRESS_CALLBACK)(void* ctx, int result, int step);
#else
typedef void (*CAML_PROGRESS_CALLBACK)(void* ctx, int result, int step);
#endif

/*
* Fixed pattern Noise Correction & Flat Field Correction
*/
CAMLINK_API(void) CamL_EnableFpnc(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsFpncEnabled(HCamL h);
CAMLINK_API(void) CamL_SetFpncThreshold(HCamL h, int value);
CAMLINK_API(int)  CamL_GetFpncThreshold(HCamL h);
CAMLINK_API(void) CamL_GetFpncThresholdRange(int* pMin, int* pMax, int* pDef);
CAMLINK_API(bool) CamL_WriteFpncData(HCamL h, int index, const unsigned char* pData, int length, CAML_PROGRESS_CALLBACK fun, void* ctx);
CAMLINK_API(bool) CamL_ReadFpncData(HCamL h, int index, unsigned char* pData, int length, CAML_PROGRESS_CALLBACK fun, void* ctx);
CAMLINK_API(void) CamL_EnableAutoFpnc(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsAutoFpncEnable(HCamL h);
CAMLINK_API(void) CamL_SetManualFpncIndex(HCamL h, int index);
CAMLINK_API(int)  CamL_GetManualFpncIndex(HCamL h);
CAMLINK_API(void) CamL_GetFpncIndexRange(HCamL h, int* pMin, int* pMax, int* pDef);

CAMLINK_API(void) CamL_EnableFfc(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsFfcEnabled(HCamL h);
CAMLINK_API(void) CamL_EnableAutoFfc(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsAutoFfcEnable(HCamL h);
CAMLINK_API(void) CamL_SetManualFfcIndex(HCamL h, int index);
CAMLINK_API(int)  CamL_GetManualFfcIndex(HCamL h);
CAMLINK_API(void) CamL_GetFfcIndexRange(HCamL h, int* pMin, int* pMax, int* pDef);
CAMLINK_API(bool) CamL_WriteFfcData(HCamL h, int index, const unsigned char* pData, int length, CAML_PROGRESS_CALLBACK fun, void* ctx);

CAMLINK_API(void) CamL_EnableDfc(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsDfcEnabled(HCamL h);
CAMLINK_API(void) CamL_SetDfcThreshold(HCamL h, int value);
CAMLINK_API(int)  CamL_GetDfcThreshold(HCamL h);
CAMLINK_API(void) CamL_GetDfcThresholdRange(int* pMin, int* pMax, int* pDef);
CAMLINK_API(bool) CamL_WriteDfcData(HCamL h, int index, const unsigned char* pData, int length, CAML_PROGRESS_CALLBACK fun, void* ctx);
CAMLINK_API(bool) CamL_ReadDfcData(HCamL h, int index, unsigned char* pData, int length, CAML_PROGRESS_CALLBACK fun, void* ctx);
CAMLINK_API(void) CamL_EnableAutoDfc(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsAutoDfcEnable(HCamL h);
CAMLINK_API(void) CamL_SetManualDfcIndex(HCamL h, int index);
CAMLINK_API(int)  CamL_GetManualDfcIndex(HCamL h);
CAMLINK_API(void) CamL_GetDfcIndexRange(HCamL h, int* pMin, int* pMax, int* pDef);

/*
* Defective Pixel Correction
*/
CAMLINK_API(void) CamL_EnableDpc(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsDpcEnabled(HCamL h);

/*
* Denoise
*/
CAMLINK_API(void) CamL_EnableDenoise(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsDenoiseEnabled(HCamL h);
CAMLINK_API(void) CamL_SetDenoise(HCamL h, int val);
CAMLINK_API(int)  CamL_GetDenoise(HCamL h);
CAMLINK_API(void) CamL_GetDenoiseRange(int* pMin, int* pMax, int* pDef);

/*
* Sharpen
*/
CAMLINK_API(void) CamL_SetSharpen(HCamL h, int val);
CAMLINK_API(int)  CamL_GetSharpen(HCamL h);
CAMLINK_API(void) CamL_GetSharpenRange(int* pMin, int* pMax, int* pDef);

/*
* Temperature
*/
CAMLINK_API(void)  CamL_SetTargetTemperature(HCamL h, float temp);
CAMLINK_API(float) CamL_GetTargetTemperature(HCamL h);
CAMLINK_API(float) CamL_GetCurrentTemperature(HCamL h);

/*
* TEC
*/
CAMLINK_API(void) CamL_EnableTec(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsTecEnabled(HCamL h);

/*
* Fan
*/
CAMLINK_API(void) CamL_EnableFan(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsFanEnabled(HCamL h);
CAMLINK_API(void) CamL_SetFan(HCamL h, int val);
CAMLINK_API(int)  CamL_GetFan(HCamL h);
CAMLINK_API(void) CamL_GetFanRange(HCamL h, int* pMin, int* pMax, int* pDef);

/*
* Trigger
*/
CAMLINK_API(void) CamL_EnableTrigger(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsTriggerEnabled(HCamL h);

/*
* Trigger Source
* source: 0x00 => Opto-isolated input, 0x01 => GPIO0, 0x02 => GPIO1, 0x03 => Counter, 0x04 => PWM, 0x05 => Software
*/
CAMLINK_API(void) CamL_SetTriggerSource(HCamL h, int source);
CAMLINK_API(int)  CamL_GetTriggerSource(HCamL h);

/*
* Software trigger always on
*/
CAMLINK_API(void) CamL_SetSoftAlwaysEnabled(HCamL h, bool bEnabled);
CAMLINK_API(bool) CamL_IsSoftAlwaysEnabled(HCamL h);

/*
* Trigger Prohibited Time
* time range: 0 ~ 5000000us
*/
CAMLINK_API(void) CamL_SetTriggerProhibitedTime(HCamL h, int time);
CAMLINK_API(int)  CamL_GetTriggerProhibitedTime(HCamL h);

/*
* Input Activation
* value: 0x00 => Rising edge, 0x01 => Falling edge, 0x02 => Level high, 0x03 => Level low
*/
CAMLINK_API(void) CamL_SetInputActivation(HCamL h, int value);
CAMLINK_API(int)  CamL_GetInputActivation(HCamL h);

/*
* Burst Counter
* value: range: [1 ~ 65535]
*/
CAMLINK_API(void) CamL_SetBurstCounter(HCamL h, int value);
CAMLINK_API(int)  CamL_GetBurstCounter(HCamL h);

/*
* Counter Source
* value: 0x00 => Opto-isolated input, 0x01 => GPIO0, 0x02 => GPIO1
*/
CAMLINK_API(void) CamL_SetCounterSource(HCamL h, int value);
CAMLINK_API(int)  CamL_GetCounterSource(HCamL h);

/*
* Counter Value
* value: range: [1 ~ 65535]
*/
CAMLINK_API(void) CamL_SetCounterValue(HCamL h, int value);
CAMLINK_API(int)  CamL_GetCounterValue(HCamL h);

/*
* PWM Source
* value: 0x00 => Opto-isolated input, 0x01 => GPIO0, 0x02 => GPIO1
*/
CAMLINK_API(void) CamL_SetPwmSource(HCamL h, int value);
CAMLINK_API(int)  CamL_GetPwmSource(HCamL h);

/*
* IO Dir
* value: 0x00 => Input, 0x01 => Output
*/
CAMLINK_API(void) CamL_SetIoDir(HCamL h, int value);
CAMLINK_API(int)  CamL_GetIoDir(HCamL h);

/*
* Trigger Delay
* ioLineNumber: -1 => Software, 0 => Opto-isolated input, 2 => GPIO0, 3 => GPIO1
* value: Trigger Delay time in microseconds, range: [0, 5000000]
*/
CAMLINK_API(void) CamL_SetTriggerDelay(HCamL h, int ioLineNumber, int value);
CAMLINK_API(int)  CamL_GetTriggerDelay(HCamL h, int ioLineNumber);

/*
* Output Mode
* ioLineNumber: 1 => Opto-isolated output, 2 => GPIO0, 3 => GPIO1
* value: 0x00 => Frame Trigger Wait, 0x01 => Exposure Active, 0x02 => Strobe, 0x03 => User output, 0x04 => Counter Output, 0x05 => Timer Output
*/
CAMLINK_API(void) CamL_SetOutputMode(HCamL h, int ioLineNumber, int value);
CAMLINK_API(int)  CamL_GetOutputMode(HCamL h, int ioLineNumber);

/*
* Strobe Duration
* Strobe duration time in microseconds, range: [0, 5000000]
*/
CAMLINK_API(void) CamL_SetStrobDuration(HCamL h, int value);
CAMLINK_API(int)  CamL_GetStrobDuration(HCamL h);

/*
* Strobe Pre Delay
* Strobe pre-delay time in microseconds, range: [0, 5000000]
*/
CAMLINK_API(void) CamL_SetStrobPreDelay(HCamL h, int value);
CAMLINK_API(int)  CamL_GetStrobPreDelay(HCamL h);

/*
* Strobe Delay
* Strobe delay time in microseconds, range: [0, 5000000]
*/
CAMLINK_API(void) CamL_SetStrobDelay(HCamL h, int value);
CAMLINK_API(int)  CamL_GetStrobDelay(HCamL h);

/*
* User Value
* value: 0 => Opto-isolated output, 1 => GPIO0 output, 2 => GPIO1 output
*/
CAMLINK_API(void) CamL_SetUserValue(HCamL h, int value);
CAMLINK_API(int)  CamL_GetUserValue(HCamL h);

/*
* Reset Counter
*/
CAMLINK_API(void) CamL_ResetCounter(HCamL h);

/*
* Debouncer Time
* ioLineNumber: 0 => Opto-isolated input, 2 => GPIO0, 3 => GPIO1
* value: debouncer time in microseconds, range: [0, 20000]
*/
CAMLINK_API(void) CamL_SetDebouncerTime(HCamL h, int ioLineNumber, int value);
CAMLINK_API(int)  CamL_GetDebouncerTime(HCamL h, int ioLineNumber);

/*
* Output Inverter
* ioLineNumber: 1 => Opto-isolated output, 2 => GPIO0, 3 => GPIO1
* bInverter: only support output signal
*/
CAMLINK_API(void) CamL_EnableOutputInverter(HCamL h, int ioLineNumber, bool bInverter);
CAMLINK_API(bool) CamL_IsOutputInverter(HCamL h, int ioLineNumber);

/*
* Output Counter Value
* value: range: [0 ~ 65535]
*/
CAMLINK_API(void) CamL_SetOutputCounterValue(HCamL h, int value);
CAMLINK_API(int)  CamL_GetOutputCounterValue(HCamL h);

/*
* Output Pause
* value: 1 => puase, 0 => unpause
*/
CAMLINK_API(void) CamL_SetOutputPause(HCamL h, int value);
CAMLINK_API(int)  CamL_GetOutputPause(HCamL h);

/*
* Soft Trigger
*/
CAMLINK_API(void) CamL_Trigger(HCamL h, bool bStart);

/*
* Clear Counter
*/
CAMLINK_API(void) CamL_ClearFrameCounter(HCamL h);
CAMLINK_API(void) CamL_ClearTriggerCounter(HCamL h);
CAMLINK_API(void) CamL_ClearTimeStampCounter(HCamL h);
CAMLINK_API(void) CamL_ClearAllCounter(HCamL h);

/*
* Enable Counter
*/
CAMLINK_API(void) CamL_EnableCounte(HCamL h, bool bEnable);
CAMLINK_API(bool) CamL_IsCounteEnabled(HCamL h);

/*
* Nuc Control
*/
CAMLINK_API(void) CamL_SetNucCtrl(HCamL h, int fpnc_en, int bias_en, int denoise_en);
CAMLINK_API(int)  CamL_GetNucCtrl(HCamL h);

/*
* Flip
*/
CAMLINK_API(void) CamL_SetVFlip(HCamL h, bool bVFlip);
CAMLINK_API(bool) CamL_IsVFlip(HCamL h);
CAMLINK_API(void) CamL_SetHFlip(HCamL h, bool bHFlip);
CAMLINK_API(bool) CamL_IsHFlip(HCamL h);

/*
* Work Mode
* mode: 0 = normal, 1 = 80bit
*/
CAMLINK_API(bool) CamL_IsSupportWorkMode(HCamL h);
CAMLINK_API(void) CamL_SetWorkMode(HCamL h, int mode);
CAMLINK_API(int)  CamL_GetWorkMode(HCamL h);

/*
* dampTime, dampGain: thousandths, range: [0, 999]
* policy: 0 = Exposure Only, 1 = Exposure Preferred, 2 = Gain Only, 3 = Gain Preferred
* peakPercent: range [0, 99], 0 = disable peak mode
*/
CAMLINK_API(bool) CamL_SetAutoExposure(HCamL h, unsigned minTime, unsigned maxTime, unsigned dampTime, unsigned minGain, unsigned maxGain, unsigned dampGain, unsigned policy, unsigned peakPercent, unsigned overPolicy);
CAMLINK_API(bool) CamL_GetAutoExposure(HCamL h, unsigned* minTime, unsigned* maxTime, unsigned* dampTime, unsigned* minGain, unsigned* maxGain, unsigned* dampGain, unsigned* policy, unsigned* peakPercent, unsigned* overPolicy);
#define CAML_AETARGET_MIN	16
#define CAML_AETARGET_MAX	220
#define CAML_AETARGET_DEF	120
CAMLINK_API(bool) CamL_SetAutoExposureTarget(HCamL h, unsigned target);
CAMLINK_API(bool) CamL_GetAutoExposureTarget(HCamL h, unsigned* target);
CAMLINK_API(bool) CamL_SetAutoExposureEnable(HCamL h, unsigned enable);
CAMLINK_API(bool) CamL_GetAutoExposureEnable(HCamL h, unsigned* enable);
CAMLINK_API(bool) CamL_AutoExposure(HCamL h, const void* data, unsigned width, unsigned height, unsigned stride, unsigned bitdepth, unsigned aeRoiLeft, unsigned aeRoiTop, unsigned aeRoiWidth, unsigned aeRoiHeight);

/*
* Bitdepth: 8~16
*/
#define CAML_BITDEPTH_MIN	8
#define CAML_BITDEPTH_MAX	16
CAMLINK_API(bool) CamL_SetBitdepth(HCamL h, int bitdepth);
CAMLINK_API(int)  CamL_GetBitdepth(HCamL h);
CAMLINK_API(bool) CamL_IsSupportBitdepth(HCamL h, int bitdepth);

/*
* Bin:       1~200
* bin < 100: bin x bin
* bin > 100: (bin / 100) x (bin % 100)
*/
#define CAML_BIN_MIN	1
#define CAML_BIN_MAX	1000
CAMLINK_API(bool) CamL_SetBin(HCamL h, int bin);
CAMLINK_API(int)  CamL_GetBin(HCamL h);
CAMLINK_API(bool) CamL_IsSupportBin(HCamL h, int bit);
CAMLINK_API(void) CamL_GetBinSize(HCamL h, int bin, int* pWidth, int* pHeight);

/*
* Bin Mode: 0 = average, 1 = saturating add
*/
CAMLINK_API(bool) CamL_SetBinMode(HCamL h, int mode);
CAMLINK_API(int)  CamL_GetBinMode(HCamL h);

/*
* Output Image Mode: 
* 0 = Sensor Image, 1 = DDR1 Image, 2 = DDR2 Image, 
* 3 = DDR1 Corrected Image, 4 = DDR2 Corrected Image, 5 = Both Corrected Image
*/
CAMLINK_API(bool) CamL_IsSupportOutputImage(HCamL h);
CAMLINK_API(bool) CamL_SetOutputImage(HCamL h, int mode);
CAMLINK_API(int)  CamL_GetOutputImage(HCamL h);

#define CAML_FEATURE_BINSKIP			0x01  /* bin/skip mode */
#define CAML_FEATURE_HIGH_FULLWELL      0x02  /* high fullwell capacity */
#define CAML_FEATURE_LOW_NOISE          0x03  /* low noise mode (Higher signal noise ratio, lower frame rate) */
#define CAML_FEATURE_GLOBAL_RESET       0x04  /* global reset mode */
#define CAML_FEATURE_CDS_MODE           0x05  /* CDS Mode: 0 = off, 1 = on, same as CamL_SetCdsMode */

CAMLINK_API(bool) CamL_SetFeature(HCamL h, int feature, int val);
CAMLINK_API(int)  CamL_GetFeature(HCamL h, int feature);
CAMLINK_API(bool) CamL_IsSupportFeature(HCamL h, int feature);

CAMLINK_API(void) CamL_SetTReady(HCamL h, int t_ready);
CAMLINK_API(int)  CamL_GetTReady(HCamL h);

CAMLINK_API(void) CamL_SetTcds(HCamL h, int tcds);
CAMLINK_API(int)  CamL_GetTcds(HCamL h);
CAMLINK_API(void) CamL_GetTcdsRange(HCamL h, int* pMin, int* pMax, int* pDef);

/* CDS MODE: 0 = off, 1 = on, same as CAML_FEATURE_CDS_MODE */
CAMLINK_API(void) CamL_SetCdsMode(HCamL h, int cds_mode);
CAMLINK_API(int)  CamL_GetCdsMode(HCamL h);

/* Parameter */
CAMLINK_API(bool) CamL_SaveParameterToCamera(HCamL h);
CAMLINK_API(void) CamL_LoadParameterFromCamera(HCamL h);
CAMLINK_API(void) CamL_FactorySettingParameter(HCamL h);

#if defined(_WIN32)
CAMLINK_API(bool) CamL_Update(HCamL h, const wchar_t* ufwFilePath, CAML_PROGRESS_CALLBACK fun, void* ctx);
#else
CAMLINK_API(bool) CamL_Update(HCamL h, const char* ufwFilePath, CAML_PROGRESS_CALLBACK fun, void* ctx);
#endif

CAMLINK_API(bool) CamL_WriteXml(HCamL h, const wchar_t* srcPath, const wchar_t* zipPath, CAML_PROGRESS_CALLBACK fun, void* ctx);
CAMLINK_API(bool) CamL_ReadXml(HCamL h, unsigned char* pData, int length, CAML_PROGRESS_CALLBACK fun, void* ctx);

CAMLINK_API(bool) CamL_IsSupportDDR(HCamL h);
CAMLINK_API(bool) CamL_WriteDDR(HCamL h, unsigned addr, const unsigned char* data, int length, CAML_PROGRESS_CALLBACK fun, void* ctx);

CAMLINK_API(bool) CamL_IsSupportEMMC(HCamL h);
CAMLINK_API(bool) CamL_WriteEMMC(HCamL h, unsigned addr, const unsigned char* data, int length, CAML_PROGRESS_CALLBACK fun, void* ctx);
CAMLINK_API(bool) CamL_ReadEMMC(HCamL h, unsigned addr, unsigned char* data, int length, CAML_PROGRESS_CALLBACK fun, void* ctx);

/* sensor taps */
CAMLINK_API(bool) CamL_IsSupportBout(HCamL h);
CAMLINK_API(void) CamL_SetBoutNumber(HCamL h, int val);
CAMLINK_API(int) CamL_GetBoutNumber(HCamL h);

/* cameralink cable number */
CAMLINK_API(bool) CamL_IsSupportCable(HCamL h);
CAMLINK_API(void) CamL_SetCableNumber(HCamL h, int val);
CAMLINK_API(int) CamL_GetCableNumber(HCamL h);

#define CAML_PORT_VR			0x01  /* external input, 3V~3.3V*/
#define CAML_PORT_SUBPV			0x02  /* external input, 3.2V~3.5V*/
#define CAML_PORT_VBLM			0x03  /* internal generation, supports external regulation, 0V~2.0V*/
CAMLINK_API(bool) CamL_SetVoltage(HCamL h, int port, int cg, int val);
CAMLINK_API(int)  CamL_GetVoltage(HCamL h, int port, int cg);
CAMLINK_API(bool) CamL_IsSupportVoltage(HCamL h, int port);

CAMLINK_API(bool) CamL_SetShutter(HCamL h, int bEnable);
CAMLINK_API(int)  CamL_GetShutter(HCamL h);
CAMLINK_API(bool) CamL_IsSupportShutter(HCamL h);
#ifdef __cplusplus
}
#endif
#endif
